/*
 * This file is part of  the extension: Ebla Multiple Link Pro
 * Copyright (c) Eblasoft Bilişim Ltd.
 *
 * This Software is the property of Eblasoft Bilişim Ltd. and is protected
 * by copyright law - it is NOT Freeware and can be used only in one project
 * under a proprietary license, which is delivered along with this program.
 * If not, see <http://eblasoft.com.tr/eula>.
 *
 * This Software is distributed as is, with LIMITED WARRANTY AND LIABILITY.
 * Any unauthorised use of this Software without a valid license is
 * a violation of the License Agreement.
 *
 * According to the terms of the license you shall not resell, sublicense,
 * rent, lease, distribute or otherwise transfer rights or usage of this
 * Software or its derivatives. You may modify the code of this Software
 * for your own needs, if source code is provided.
 */

define('ebla-link-pro:views/fields/link-multiple-form/inline-list', 'views/record/list', function (Dep) {

    return Dep.extend({

        rowActionsView: 'ebla-link-pro:views/record/row-actions',

        _getHeaderDefs: function () {
            let headerDefs = Dep.prototype._getHeaderDefs.call(this);

            const forbiddenFieldList = this.getForbiddenFields();

            headerDefs = headerDefs.filter((item) => !forbiddenFieldList.includes(item.name));

            if (this.options.mode === 'edit') {
                headerDefs.forEach((hDef) => {
                    const name = hDef.name;

                    if (!name) return;

                    hDef.isSortable = false;

                    const required = this.requiredFields[name] || this.getMetadata().get([
                        'entityDefs',
                        this.collection.entityType,
                        'fields',
                        name,
                        'required'
                    ]);

                    hDef.label = hDef.label || '';

                    required && (hDef.label += ' *');
                });
            }

            return headerDefs;
        },

        _convertLayout: function (listLayout, model) {
            const forbiddenFieldList = this.getForbiddenFields();

            listLayout = listLayout.filter((item) => !forbiddenFieldList.includes(item.name));

            return Dep.prototype._convertLayout.call(this, listLayout, model);
        },

        getItemEl: function (model, item) {
            return this.options.el + ' tr[data-id="' + model.id + '"] td.cell[data-name="' + item.columnName + '"] .cell-wrapper';
        },

        getForbiddenFields: function () {
            this._cachedScopeForbiddenFieldList = this._cachedScopeForbiddenFieldList ||
                this.getAcl().getScopeForbiddenFieldList(this.entityType, 'read');
            return this._cachedScopeForbiddenFieldList;
        },

        getReadOnlyFields: function () {
            if (!this._cachedScopeReadOnlyFieldList) {
                this._cachedScopeReadOnlyFieldList = (
                    this.getAcl().getScopeForbiddenFieldList(this.entityType, 'edit').concat(
                        this.options.readOnlyFields
                    )
                );
            }

            return this._cachedScopeReadOnlyFieldList;
        },

        prepareInternalLayout: function (internalLayout, model) {
            const readOnlyFieldList = this.getReadOnlyFields();

            internalLayout.forEach((item, i) => {
                item.el = this.getItemEl(model, item);

                if (item.columnName === 'buttons' && i === internalLayout.length - 1) return;
                if (this.options.mode === 'edit' && !readOnlyFieldList.includes(item.columnName) && this.getAcl().checkModel(model, 'edit')) {
                    item.options.mode = this.options.mode;
                }
                item.tag = 'div';
                item.class = 'cell-wrapper';
            });
        },

        processLinkClick(id) {
            this.actionQuickView({id});
        },

        setup: function () {
            Dep.prototype.setup.call(this);
            this.model = this.options.model;
            this.requiredFields = this.options.requiredFields || {};
        },

        afterRender: function () {
            Dep.prototype.afterRender.call(this);
        },

        actionQuickEdit: function (data) {
            let model = this.collection.get(data?.id);
            // save model if changed
            if (model && model.hasChanged()) {
                model.save().then(() => {
                    Dep.prototype.actionQuickEdit.call(this, _.extend(data, {
                        noFullForm: this.options.noFullForm
                    }));
                });
            } else {
                Dep.prototype.actionQuickEdit.call(this, _.extend(data, {
                    noFullForm: this.options.noFullForm
                }));
            }
        },

        actionQuickView: function (data) {
            data = data || {};
            const id = data.id;
            if (!id) return;

            let model = null;
            if (this.collection) {
                model = this.collection.get(id);
            }
            if (!data.scope && !model) {
                return;
            }

            const scope = data.scope || model.name || this.scope;

            const viewName = this.getMetadata().get('clientDefs.' + scope + '.modalViews.detail') || 'views/modals/detail';

            if (!this.quickDetailDisabled) {
                Espo.Ui.notify(this.translate('loading', 'messages'));

                const options = {
                    scope: scope,
                    model: model,
                    fullFormDisabled: this.options.noFullForm,
                    id: id,
                    quickEditDisabled: this.quickEditDisabled,
                };
                if (this.options.keepCurrentRootUrl) {
                    options.rootUrl = this.getRouter().getCurrentUrl();
                }
                this.createView('modal', viewName, options, function (view) {
                    this.listenToOnce(view, 'after:render', function () {
                        Espo.Ui.notify(false);
                    });
                    view.render();

                    this.listenToOnce(view, 'remove', function () {
                        this.clearView('modal');
                    }, this);

                    this.listenToOnce(view, 'after:edit-cancel', function () {
                        this.actionQuickView({id: view.model.id, scope: view.model.name});
                    }, this);

                    this.listenToOnce(view, 'after:save', function (model) {
                        this.trigger('after:save', model);
                    }, this);
                }, this);
            } else {
                this.getRouter().navigate('#' + scope + '/view/' + id, {trigger: true});
            }
        },

        actionUnlinkRelated: function (data) {
            const id = data.id;

            if (this.collection.get(id).isNewItem) {
                return this.removeRecordFromList(id);
            }

            this.confirm({
                message: this.translate('unlinkRecordConfirmation', 'messages'),
                confirmText: this.translate('Unlink')
            }, function () {
                const model = this.collection.get(id);
                this.notify('Unlinking...');
                Espo.Ajax.deleteRequest('EblaUnlink/' + this.collection.url, {
                    id: id
                }).then(function () {
                    this.notify('Unlinked', 'success');
                    this.removeRecordFromList(id);
                    this.model.trigger('after:unrelate');
                    this.trigger('after:unrelate', {inlineList: true, id});
                    this.getParentView()?.deleteLink(id);
                }.bind(this));
            }, this);
        },

        actionRemoveRelated: function (data) {
            const {id} = data || {};

            if (!id) {
                return;
            }

            const model = this.collection.get(id);

            if (model.isNewItem) {
                return this.removeRecordFromList(id);
            }

            if (!this.getAcl().checkModel(model, 'delete')) {
                this.notify('Access denied', 'error');

                return false;
            }

            this.confirm({
                message: this.translate('removeRecordConfirmation', 'messages', this.entityType),
                confirmText: this.translate('Remove'),
            }, () => {
                this.collection.trigger('model-removing', id);
                this.collection.remove(model);

                this.notify('Removing...');

                model.destroy({wait: true})
                    .then(() => {
                        this.notify('Removed', 'success');
                        this.trigger('after:inline-remove', {id});
                        this.removeRecordFromList(id);
                        this.getParentView()?.deleteLink(id);
                    })
                    .catch(() => {
                        this.notify('Error occurred', 'error');
                        this.collection.push(model);
                    });
            });
        },

        removeRecordFromList: function (id) {
            Dep.prototype.removeRecordFromList.call(this, id);
            this.getParentView()?.fetchToModel();
            this.getParentView()?.trigger('change');
        },
    });
});
