<?php
/***********************************************************************************
 * The contents of this file are subject to the Extension License Agreement
 * ("Agreement") which can be viewed at
 * https://www.espocrm.com/extension-license-agreement/.
 * By copying, installing downloading, or using this file, You have unconditionally
 * agreed to the terms and conditions of the Agreement, and You may not use this
 * file except in compliance with the Agreement. Under the terms of the Agreement,
 * You shall not license, sublicense, sell, resell, rent, lease, lend, distribute,
 * redistribute, market, publish, commercialize, or otherwise transfer rights or
 * usage to the software or any modified version or derivative work of the software
 * created by or for you.
 *
 * Copyright (C) 2015-2025 Letrium Ltd.
 *
 * License ID: e4c270586a0c8a9fda53bda910f357e0
 ************************************************************************************/

namespace Espo\Modules\Advanced\Core\Workflow;

use DateTime;
use DateTimeZone;
use Espo\Core\ORM\Entity as CoreEntity;
use Espo\Core\Utils\DateTime as DateTimeUtil;
use Espo\ORM\Entity;
use Espo\ORM\EntityManager;
use RuntimeException;
use stdClass;

class Utils
{
    /**
     * String to lower case.
     * @todo Revise. Remove?
     */
    public static function strtolower(?string $str): ?string
    {
        if (!empty($str)) {
            return mb_strtolower($str, 'UTF-8');
        }

        return $str;
    }

    /**
     * Shift date days.
     *
     * @param int $shiftDays
     * @param ?string $input
     * @param 'datetime'|'date' $type
     * @param string $unit
     * @param ?string $timezone
     * @return string
     */
    public static function shiftDays(
        $shiftDays = 0,
        $input = null,
        $type = 'datetime',
        $unit = 'days',
        $timezone = null
    ): string {

        if (!in_array($unit, ['hours', 'minutes', 'days', 'months'])) {
            throw new RuntimeException("Not supported date shift interval unit $unit.");
        }

        $dateTime = new DateTime($input);
        $dateTime->setTimezone(new DateTimeZone($timezone ?? 'UTC'));

        if ($type === 'date') {
            $dateTime->setTime(0, 0);
        }

        if ($shiftDays) {
            $dateTime->modify("$shiftDays $unit");
        }

        if ($type === 'datetime') {
            $dateTime->setTimezone(new DateTimeZone('UTC'));

            return $dateTime->format(DateTimeUtil::SYSTEM_DATE_TIME_FORMAT);
        }

        $dateTime->setTime(0, 0);

        return $dateTime->format(DateTimeUtil::SYSTEM_DATE_FORMAT);
    }

    /**
     * @deprecated
     *
     * Get field value for a field/related field. If this field has a relation, get value from the relation.
     *
     * @param ?string $fieldName
     * @param bool $returnEntity
     * @param ?EntityManager $entityManager
     * @param ?stdClass $createdEntitiesData
     * @return mixed
     */
    public static function getFieldValue(
        CoreEntity $entity,
        $fieldName,
        $returnEntity = false,
        $entityManager = null,
        $createdEntitiesData = null
    ): mixed {

        if (str_starts_with($fieldName, 'created:')) {
            [$alias, $field] = explode('.', substr($fieldName, 8));

            if (!$createdEntitiesData) {
                return null;
            }

            if (!isset($createdEntitiesData->$alias)) {
                return null;
            }

            $entityTypeValue = $createdEntitiesData->$alias->entityType ?? null;
            $entityIdValue = $createdEntitiesData->$alias->entityId ?? null;

            if (!$entityTypeValue || !$entityIdValue) {
                return null;
            }

            $entity = $entityManager->getEntityById($entityTypeValue, $entityIdValue);

            if (!$entity) {
                return null;
            }

            $fieldName = $field;
        } else if (str_contains($fieldName, '.')) {
            [$first, $foreignName] = explode('.', $fieldName);

            $relatedEntity = null;

            if (
                $entityManager &&
                $entity->hasRelation($first) &&
                in_array($entity->getRelationType($first), [
                    Entity::BELONGS_TO,
                    Entity::HAS_ONE,
                    Entity::BELONGS_TO_PARENT,
                ])
            ) {
                $relatedEntity = $entityManager
                    ->getRDBRepository($entity->getEntityType())
                    ->getRelation($entity, $first)
                    ->findOne();
            }

            // If the entity is just created and doesn't have added relations.
            if (
                $entityManager &&
                !$relatedEntity &&
                $entity->hasRelation($first)
            ) {
                $foreignEntityType = $entity->getRelationParam($first, 'entity');

                $firstIdAttr = static::normalizeFieldName($entity, $first);

                if (
                    $foreignEntityType &&
                    $entity->hasAttribute($firstIdAttr) &&
                    $entity->get($firstIdAttr)
                ) {
                    $relatedEntity = $entityManager->getEntityById($foreignEntityType, $entity->get($firstIdAttr));
                }
            }

            if ($relatedEntity instanceof CoreEntity) {
                $entity = $relatedEntity;

                $fieldName = $foreignName;
            } else {
                $GLOBALS['log']->error("Workflow: Could not get related entity for '$fieldName'.");

                return null;
            }
        }

        if (!$entity instanceof CoreEntity) {
            throw new RuntimeException();
        }

        if ($entity->hasRelation($fieldName)) {
            $relatedEntity = null;

            if ($entity->getRelationType($fieldName) === Entity::BELONGS_TO_PARENT) {
                $valueType = $entity->get($fieldName . 'Type');
                $valueId = $entity->get($fieldName . 'Id');

                if ($valueType && $valueId) {
                    $relatedEntity = $entityManager->getEntityById($valueType, $valueId);
                }
            } else if (
                in_array($entity->getRelationType($fieldName), [Entity::BELONGS_TO, Entity::HAS_ONE])
            ) {
                $relatedEntity = $entityManager
                    ->getRDBRepository($entity->getEntityType())
                    ->getRelation($entity, $fieldName)
                    ->findOne();
            }

            if ($relatedEntity instanceof CoreEntity) {
                $foreignKey = Utils::getRelationOption($entity, 'foreignKey', $fieldName, 'id');

                return $returnEntity ? $relatedEntity : $relatedEntity->get($foreignKey);
            }

            if (!$relatedEntity) {
                $normalizedFieldName = static::normalizeFieldName($entity, $fieldName);

                if (!$entity->isNew()) {
                    if ($entity->hasLinkMultipleField($fieldName)) {
                        $entity->loadLinkMultipleField($fieldName);
                    }
                }

                if ($entity->getRelationType($fieldName) === Entity::BELONGS_TO_PARENT && !$returnEntity) {
                    return null;
                }

                $fieldValue = $returnEntity ?
                    static::getParentEntity($entity, $fieldName, $entityManager) :
                    static::getParentValue($entity, $normalizedFieldName);

                if (isset($fieldValue)) {
                    return $fieldValue;
                }
            }

            if ($entity instanceof CoreEntity && $entity->hasLinkMultipleField($fieldName)) {
                $entity->loadLinkMultipleField($fieldName);
            }

            return $returnEntity ? null : $entity->get($fieldName . 'Ids');
        }

        switch ($entity->getAttributeType($fieldName)) {
            // @todo Revise.
            case 'linkParent':
                $fieldName .= 'Id';

                break;
        }

        if ($returnEntity) {
            return $entity;
        }

        if ($entity->hasAttribute($fieldName)) {
            return $entity->get($fieldName);
        }

        return null;
    }

    /**
     * Get parent field value. Works for parent and regular fields,
     *
     * @param string|string[] $normalizedFieldName
     * @return mixed
     */
    public static function getParentValue(Entity $entity, $normalizedFieldName)
    {
        if (is_array($normalizedFieldName)) {
            $value = [];

            foreach ($normalizedFieldName as $fieldName) {
                if ($entity->hasAttribute($fieldName)) {
                    $value[$fieldName] = $entity->get($fieldName);
                }
            }

            return $value;
        }

        if ($entity->hasAttribute($normalizedFieldName)) {
            return $entity->get($normalizedFieldName);
        }

        return null;
    }

    /**
     * @param CoreEntity $entity
     * @param string $fieldName
     * @param ?EntityManager $entityManager
     * @return CoreEntity|Entity|null
     */
    public static function getParentEntity(CoreEntity $entity, string $fieldName, $entityManager = null)
    {
        if (!$entity->hasRelation($fieldName)) {
            return $entity;
        }

        if ($entityManager instanceof EntityManager) {
            $normalizedFieldName = static::normalizeFieldName($entity, $fieldName);

            $fieldValue = static::getParentValue($entity, $normalizedFieldName);

            if (isset($fieldValue) && is_string($fieldValue)) {
                $fieldEntityDefs = $entityManager->getMetadata()->get($entity->getEntityType());

                if (isset($fieldEntityDefs['relations'][$fieldName]['entity'])) {
                    $fieldEntity = $fieldEntityDefs['relations'][$fieldName]['entity'];

                    return $entityManager->getEntityById($fieldEntity, $fieldValue);
                }
            }
        }

        return null;
    }

    /**
     * @param string $field
     * @return string
     * @deprecated Use getActualAttributes in Helper.
     * Normalize field name for fields and relations.
     */
    public static function normalizeFieldName(CoreEntity $entity, $field)
    {
        if ($entity->hasRelation($field)) {
            $type = $entity->getRelationType($field);

            $key = $entity->getRelationParam($field, 'key');

            switch ($type) {
                case 'belongsTo':
                    if ($key) {
                        $field = $key;
                    }

                    break;

                case 'belongsToParent':
                    $field = [
                        $field . 'Id',
                        $field . 'Type',
                    ];

                    break;

                case 'hasChildren':
                case 'hasMany':
                case 'manyMany':
                    $field .= 'Ids';

                    break;
            }

            return $field;
        }

        if ($entity->hasAttribute($field . 'Id')) {
            $fieldType = $entity->getAttributeParam($field . 'Id', 'fieldType');

            if ($fieldType === 'link' || $fieldType === 'linkParent') {
                $field = $field . 'Id';
            }
        }

        return $field;
    }

    /**
     * Get option value for the relation.
     *
     * @param string $optionName
     * @param string $relationName
     * @param mixed $returns
     * @return mixed
     */
    public static function getRelationOption(CoreEntity $entity, $optionName, $relationName, $returns = null)
    {
        if (!$entity->hasRelation($relationName)) {
            return $returns;
        }

        return $entity->getRelationParam($relationName, $optionName) ?? $returns;
    }

    public static function getAttributeType(CoreEntity $entity, string $name): ?string
    {
        if (!$entity->hasAttribute($name)) {
            $name = static::normalizeFieldName($entity, $name);

            if (!is_string($name)) {
                return null;
            }
        }

        return $entity->getAttributeType($name);
    }
}
