<?php

namespace Espo\Modules\EblaKanbanMultiple\Tools\Kanban;

use Espo\Core\Exceptions\Error;
use Espo\Core\FieldProcessing\ListLoadProcessor;
use Espo\Core\FieldProcessing\Loader\Params as FieldLoaderParams;
use Espo\Core\Record\Collection;
use Espo\Core\Record\ServiceContainer as RecordServiceContainer;
use Espo\Core\Select\SearchParams;
use Espo\Core\Select\SelectBuilderFactory;
use Espo\Core\Utils\Metadata;
use Espo\ORM\Entity;
use Espo\ORM\EntityManager;
use Espo\Tools\Kanban\GroupItem;
use Espo\Tools\Kanban\Kanban;
use Espo\Tools\Kanban\Result;

class EblaKanban extends Kanban
{
    private $metadata;
    private $entityManager;
    private $listLoadProcessor;
    private const MAX_GROUP_LENGTH = 100;
    protected ?string $entityType = null;
    protected ?SearchParams $searchParams = null;
    private $recordServiceContainer;
    private $selectBuilderFactory;
    protected ?string $userId = null;
    private const DEFAULT_MAX_ORDER_NUMBER = 50;
    protected int $maxOrderNumber = self::DEFAULT_MAX_ORDER_NUMBER;

    public function __construct(Metadata $metadata, SelectBuilderFactory $selectBuilderFactory, EntityManager $entityManager, ListLoadProcessor $listLoadProcessor, RecordServiceContainer $recordServiceContainer)
    {
        parent::__construct($metadata, $selectBuilderFactory, $entityManager, $listLoadProcessor, $recordServiceContainer);
        $this->metadata = $metadata;
        $this->entityManager = $entityManager;
        $this->listLoadProcessor = $listLoadProcessor;
        $this->recordServiceContainer = $recordServiceContainer;
        $this->selectBuilderFactory = $selectBuilderFactory;
    }

    public function setEntityType(string $entityType): self
    {
        $this->entityType = $entityType;
        return $this;
    }

    public function setCountDisabled(bool $countDisabled): self
    {
        $this->countDisabled = $countDisabled;
        return $this;
    }

    public function setOrderDisabled(bool $orderDisabled): self
    {
        $this->orderDisabled = $orderDisabled;
        return $this;
    }

    public function setSearchParams(SearchParams $searchParams): self
    {
        $this->searchParams = $searchParams;
        return $this;
    }

    public function setMaxOrderNumber(?int $maxOrderNumber): self
    {
        if ($maxOrderNumber === null) {
            $this->maxOrderNumber = self::DEFAULT_MAX_ORDER_NUMBER;
            return $this;
        }
        $this->maxOrderNumber = $maxOrderNumber;
        return $this;
    }

    /**
     * @throws Error
     */
    protected function getStatusField(): string
    {

        if (empty($_REQUEST['statusField'])) {
            assert(is_string($this->entityType));
            $statusField = $this->metadata->get(['scopes', $this->entityType, 'statusField']);

            if (!$statusField) {
                throw new Error("No status field for entity type '{$this->entityType}'.");
            }

            return $statusField;
        } else {
            return $_REQUEST['statusField'];
        }
    }

    public function getResult(): Result
    {
        if (!$this->entityType) {
            throw new Error("Entity type is not specified.");
        }

        if (!$this->searchParams) {
            throw new Error("No search params.");
        }

        $searchParams = $this->searchParams;

        $recordService = $this->recordServiceContainer->get($this->entityType);

        $maxSize = $searchParams->getMaxSize();

        if ($this->countDisabled && $maxSize) {
            $searchParams = $searchParams->withMaxSize($maxSize + 1);
        }

        $query = $this->selectBuilderFactory
            ->create()
            ->from($this->entityType)
            ->withStrictAccessControl()
            ->withSearchParams($searchParams)
            ->build();

        $statusField = $this->getStatusField();
        $statusList = $this->getStatusList();
        $statusIgnoreList = $this->getStatusIgnoreList();

        $groupList = [];

        $repository = $this->entityManager->getRDBRepository($this->entityType);

        $hasMore = false;

        foreach ($statusList as $status) {
            if (in_array($status, $statusIgnoreList)) {
                continue;
            }

            if (!$status) {
                continue;
            }

            $itemSelectBuilder = $this->entityManager
                ->getQueryBuilder()
                ->select()
                ->clone($query);

            $itemSelectBuilder->where([
                $statusField => $status,
            ]);

            $itemQuery = $itemSelectBuilder->build();

            $newOrder = $itemQuery->getOrder();

            array_unshift($newOrder, [
                'COALESCE:(kanbanOrder.order, ' . ($this->maxOrderNumber + 1) . ')',
                'ASC',
            ]);

            if ($this->userId && !$this->orderDisabled) {
                $group = mb_substr($status, 0, self::MAX_GROUP_LENGTH);

                $itemQuery = $this->entityManager
                    ->getQueryBuilder()
                    ->select()
                    ->clone($itemQuery)
                    ->order($newOrder)
                    ->leftJoin(
                        'KanbanOrder',
                        'kanbanOrder',
                        [
                            'kanbanOrder.entityType' => $this->entityType,
                            'kanbanOrder.entityId:' => 'id',
                            'kanbanOrder.group' => $group,
                            'kanbanOrder.userId' => $this->userId,
                        ]
                    )
                    ->build();
            }

            $collectionSub = $repository
                ->clone($itemQuery)
                ->find();

            if (!$this->countDisabled) {
                $totalSub = $repository->clone($itemQuery)->count();
            }
            else {
                $recordCollection = Collection::createNoCount($collectionSub, $maxSize);

                $collectionSub = $recordCollection->getCollection();
                $totalSub = $recordCollection->getTotal();

                if ($totalSub === Collection::TOTAL_HAS_MORE) {
                    $hasMore = true;
                }
            }

            $loadProcessorParams = FieldLoaderParams
                ::create()
                ->withSelect($searchParams->getSelect());

            foreach ($collectionSub as $e) {
                $this->listLoadProcessor->process($e, $loadProcessorParams);

                $recordService->prepareEntityForOutput($e);
            }

            /** @var Collection<Entity> $itemRecordCollection */
            $itemRecordCollection = new Collection($collectionSub, $totalSub);

            $groupList[] = new GroupItem($status, $itemRecordCollection);
        }

        $total = !$this->countDisabled ?
            $repository->clone($query)->count() :
            ($hasMore ? Collection::TOTAL_HAS_MORE : Collection::TOTAL_HAS_NO_MORE);

        return new Result($groupList, $total);
    }

    protected function getStatusList(): array
    {
        assert(is_string($this->entityType));

        $statusField = $this->getStatusField();

        $statusList = $this->metadata->get(['entityDefs', $this->entityType, 'fields', $statusField, 'options']);

        if (empty($statusList)) {
            throw new Error("No options for status field for entity type '{$this->entityType}'.");
        }

        return $statusList;
    }

    /**
     * @throws Error
     */
    protected function getStatusIgnoreList(): array
    {
        $ignoreOptionsMap = $this->metadata->get(['clientDefs', $this->entityType, 'eblaKanbanMultipleIgnoreOptions'], []);

        $ignoreOptions = $ignoreOptionsMap[$this->getStatusField()];
        if (is_array($ignoreOptions)) {
            return $ignoreOptions;
        }

        return $this->metadata->get(['scopes', $this->entityType, 'kanbanStatusIgnoreList'], []);
    }
}
