<?php
/***********************************************************************************
 * The contents of this file are subject to the Extension License Agreement
 * ("Agreement") which can be viewed at
 * https://www.espocrm.com/extension-license-agreement/.
 * By copying, installing downloading, or using this file, You have unconditionally
 * agreed to the terms and conditions of the Agreement, and You may not use this
 * file except in compliance with the Agreement. Under the terms of the Agreement,
 * You shall not license, sublicense, sell, resell, rent, lease, lend, distribute,
 * redistribute, market, publish, commercialize, or otherwise transfer rights or
 * usage to the software or any modified version or derivative work of the software
 * created by or for you.
 *
 * Copyright (C) 2024-2025 Letrium Ltd.
 *
 * License ID: f27e70ce6801a13265271f5669c8bc5c
 ************************************************************************************/

namespace Espo\Modules\Project\Classes\Acl\ProjectTask;

use Espo\Core\Acl\AccessEntityCREDSChecker;
use Espo\Core\Acl\DefaultAccessChecker;
use Espo\Core\Acl\ScopeData;
use Espo\Entities\User;
use Espo\Modules\Project\Entities\ProjectRole;
use Espo\Modules\Project\Entities\ProjectTask;
use Espo\Modules\Project\Tools\Project\MemberRoleProvider;
use Espo\ORM\Entity;
use Espo\ORM\EntityManager;

/**
 * @implements AccessEntityCREDSChecker<ProjectTask>
 */
class AccessChecker implements AccessEntityCREDSChecker
{
    public function __construct(
        private DefaultAccessChecker $defaultAccessChecker,
        private MemberRoleProvider $memberRoleProvider,
        private EntityManager $entityManager,
    ) {}

    public function check(User $user, ScopeData $data): bool
    {
        return $this->defaultAccessChecker->check($user, $data);
    }

    public function checkCreate(User $user, ScopeData $data): bool
    {
        return $this->defaultAccessChecker->checkCreate($user, $data);
    }

    public function checkRead(User $user, ScopeData $data): bool
    {
        return $this->defaultAccessChecker->checkRead($user, $data);
    }

    public function checkEdit(User $user, ScopeData $data): bool
    {
        return $this->defaultAccessChecker->checkEdit($user, $data);
    }

    public function checkDelete(User $user, ScopeData $data): bool
    {
        return $this->defaultAccessChecker->checkDelete($user, $data);
    }

    public function checkStream(User $user, ScopeData $data): bool
    {
        return $this->defaultAccessChecker->checkStream($user, $data);
    }

    public function checkEntityCreate(User $user, Entity $entity, ScopeData $data): bool
    {
        if (!$entity->getProjectId()) {
            return false;
        }

        if ($user->isAdmin()) {
            return true;
        }

        if ($entity->getParentTaskId()) {
            $parentTask = $this->entityManager
                ->getRDBRepositoryByClass(ProjectTask::class)
                ->getById($entity->getParentTaskId());

            if (!$parentTask) {
                return false;
            }

            if (!$this->checkEntityEdit($user, $parentTask, $data)) {
                return false;
            }
        }

        $memberRole = $this->memberRoleProvider->get($user, $entity->getProjectId());

        if (!$memberRole) {
            return false;
        }

        return $memberRole->getTaskCreate() === ProjectRole::LEVEL_YES;
    }

    public function checkEntityRead(User $user, Entity $entity, ScopeData $data): bool
    {
        if ($user->getId() === $entity->getOwnerId()) {
            return true;
        }

        if ($user->getId() === $entity->getAssignedUserId()) {
            return true;
        }

        if ($user->isAdmin()) {
            return true;
        }

        if ($entity->getParentTaskId()) {
            $parentTask = $this->entityManager
                ->getRDBRepositoryByClass(ProjectTask::class)
                ->getById($entity->getParentTaskId());

            if ($parentTask && $this->checkEntityRead($user, $parentTask, $data)) {
                return true;
            }
        }

        $memberRole = $this->memberRoleProvider->get($user, $entity->getProjectId());

        if (!$memberRole) {
            return false;
        }

        if ($memberRole->getTaskRead() === ProjectRole::LEVEL_ALL) {
            return true;
        }

        return false;
    }

    public function checkEntityEdit(User $user, Entity $entity, ScopeData $data): bool
    {
        if ($user->getId() === $entity->getOwnerId()) {
            return true;
        }

        if ($user->isAdmin()) {
            return true;
        }

        if ($entity->getParentTaskId()) {
            $parentTask = $this->entityManager
                ->getRDBRepositoryByClass(ProjectTask::class)
                ->getById($entity->getParentTaskId());

            if ($parentTask && $this->checkEntityEdit($user, $parentTask, $data)) {
                return true;
            }
        }

        $memberRole = $this->memberRoleProvider->get($user, $entity->getProjectId());

        if (!$memberRole) {
            return false;
        }

        if ($memberRole->getTaskEdit() === ProjectRole::LEVEL_ALL) {
            return true;
        }

        if (
            $memberRole->getTaskEdit() === ProjectRole::LEVEL_ASSIGNED &&
            $user->getId() === $entity->getAssignedUserId()
        ) {
            return true;
        }

        return false;
    }

    public function checkEntityDelete(User $user, Entity $entity, ScopeData $data): bool
    {
        if ($user->isAdmin()) {
            return true;
        }

        if ($entity->getParentTaskId()) {
            $parentTask = $this->entityManager
                ->getRDBRepositoryByClass(ProjectTask::class)
                ->getById($entity->getParentTaskId());

            if ($parentTask && $this->checkEntityEdit($user, $parentTask, $data)) {
                return true;
            }
        }

        $memberRole = $this->memberRoleProvider->get($user, $entity->getProjectId());

        if (!$memberRole) {
            return false;
        }

        if ($memberRole->getTaskDelete() === ProjectRole::LEVEL_ALL) {
            return true;
        }

        if (
            $memberRole->getTaskDelete() === ProjectRole::LEVEL_OWN &&
            $user->getId() === $entity->getOwnerId()
        ) {
            return true;
        }

        return false;
    }

    public function checkEntityStream(User $user, Entity $entity, ScopeData $data): bool
    {
        if ($user->getId() === $entity->getOwnerId()) {
            return true;
        }

        if ($user->getId() === $entity->getAssignedUserId()) {
            return true;
        }

        if ($user->isAdmin()) {
            return true;
        }

        if ($entity->getParentTaskId()) {
            $parentTask = $this->entityManager
                ->getRDBRepositoryByClass(ProjectTask::class)
                ->getById($entity->getParentTaskId());

            if ($parentTask && $this->checkEntityStream($user, $parentTask, $data)) {
                return true;
            }
        }

        $memberRole = $this->memberRoleProvider->get($user, $entity->getProjectId());

        if (!$memberRole) {
            return false;
        }

        if ($memberRole->getTaskStream() === ProjectRole::LEVEL_ALL) {
            return true;
        }

        return false;
    }
}
