<?php
/***********************************************************************************
 * The contents of this file are subject to the Extension License Agreement
 * ("Agreement") which can be viewed at
 * https://www.espocrm.com/extension-license-agreement/.
 * By copying, installing downloading, or using this file, You have unconditionally
 * agreed to the terms and conditions of the Agreement, and You may not use this
 * file except in compliance with the Agreement. Under the terms of the Agreement,
 * You shall not license, sublicense, sell, resell, rent, lease, lend, distribute,
 * redistribute, market, publish, commercialize, or otherwise transfer rights or
 * usage to the software or any modified version or derivative work of the software
 * created by or for you.
 *
 * Copyright (C) 2024-2025 Letrium Ltd.
 *
 * License ID: f27e70ce6801a13265271f5669c8bc5c
 ************************************************************************************/

namespace Espo\Modules\Project\Select\ProjectTask\Access\Filters;

use Espo\Core\Select\AccessControl\Filter;
use Espo\Entities\User;
use Espo\Modules\Project\Entities\Project;
use Espo\Modules\Project\Entities\ProjectRole;
use Espo\ORM\EntityManager;
use Espo\ORM\Query\Part\Condition as Cond;
use Espo\ORM\Query\Part\Expression as Expr;
use Espo\ORM\Query\SelectBuilder as QueryBuilder;
use Traversable;

class Member implements Filter
{
    public function __construct(
        private User $user,
        private EntityManager $entityManager,
    ) {}

    public function apply(QueryBuilder $queryBuilder): void
    {
        /** @var Traversable<ProjectRole> $roles */
        $roles = $this->entityManager
            ->getRDBRepositoryByClass(ProjectRole::class)
            ->select('id')
            ->where(['taskRead' => ProjectRole::LEVEL_ALL])
            ->find();

        $roleIds = array_map(fn (ProjectRole $role) => $role->getId(), iterator_to_array($roles));

        $queryBuilder
            ->leftJoin('parentTask', 'parentTaskAccess')
            ->where(
                Cond::or(
                    Cond::equal(
                        Expr::column('assignedUserId'),
                        $this->user->getId(),
                    ),
                    Cond::equal(
                        Expr::column('ownerId'),
                        $this->user->getId(),
                    ),
                    Cond::in(
                        Expr::column('projectId'),
                        QueryBuilder
                            ::create()
                            ->from(Project::RELATIONSHIP_PROJECT_USER)
                            ->select('projectId')
                            ->where([
                                'userId' => $this->user->getId(),
                                'deleted' => false,
                            ])
                            ->where(
                                Cond::or(
                                    Cond::in(Expr::column('role'), [Project::ROLE_OWNER, Project::ROLE_EDITOR]),
                                    Cond::in(Expr::column('roleId'), $roleIds),
                                )
                            )
                            ->build()
                    ),
                    Cond::and(
                        Cond::notEqual(Expr::column('parentTaskId'), null),
                        Cond::or(
                            Cond::equal(
                                Expr::column('parentTaskAccess.assignedUserId'),
                                $this->user->getId(),
                            ),
                            Cond::equal(
                                Expr::column('parentTaskAccess.ownerId'),
                                $this->user->getId(),
                            ),
                        ),
                    ),
                ),
            );
    }
}
