<?php
/***********************************************************************************
 * The contents of this file are subject to the Extension License Agreement
 * ("Agreement") which can be viewed at
 * https://www.espocrm.com/extension-license-agreement/.
 * By copying, installing downloading, or using this file, You have unconditionally
 * agreed to the terms and conditions of the Agreement, and You may not use this
 * file except in compliance with the Agreement. Under the terms of the Agreement,
 * You shall not license, sublicense, sell, resell, rent, lease, lend, distribute,
 * redistribute, market, publish, commercialize, or otherwise transfer rights or
 * usage to the software or any modified version or derivative work of the software
 * created by or for you.
 *
 * Copyright (C) 2024-2025 Letrium Ltd.
 *
 * License ID: f27e70ce6801a13265271f5669c8bc5c
 ************************************************************************************/

namespace Espo\Modules\Project\Tools\Project\Api;

use Espo\Core\Api\Action;
use Espo\Core\Api\Request;
use Espo\Core\Api\Response;
use Espo\Core\Api\ResponseComposer;
use Espo\Core\Exceptions\BadRequest;
use Espo\Core\Exceptions\Error\Body;
use Espo\Core\Exceptions\Forbidden;
use Espo\Core\Exceptions\ForbiddenSilent;
use Espo\Core\Field\Date;
use Espo\Core\Record\EntityProvider;
use Espo\Modules\Project\Entities\Project;
use Espo\Modules\Project\Entities\ProjectGroup;
use Espo\Modules\Project\Entities\ProjectTask;
use Espo\Modules\Project\Tools\Project\Gantt\NonWorkingRangesService;
use Espo\Modules\Project\Tools\Project\Gantt\RangeService;
use Espo\Modules\Project\Tools\Project\Gantt\Service;
use Espo\ORM\Collection;

/**
 * @noinspection PhpUnused
 */
class GetGantt implements Action
{
    private const MAX_WORKING_CALENDAR_DAY_RANGE = 200;

    public function __construct(
        private EntityProvider $entityProvider,
        private Service $service,
        private RangeService $rangeService,
        private NonWorkingRangesService $nonWorkingRangesService,
    ) {}

    public function process(Request $request): Response
    {
        $id = $request->getRouteParam('id') ?? throw new BadRequest();
        $scaleUnit = $request->getQueryParam('scaleUnit') ?? 'week';
        $scale = (int) ($request->getQueryParam('scale') ?? '1');
        $collapsedGroupIds = explode(',', $request->getQueryParam('collapsedGroupIds') ?? '');

        if ($scaleUnit !== 'week' && $scaleUnit !== 'month') {
            throw new BadRequest("Bad rangeUnit.");
        }

        if ($scale < 1) {
            throw new BadRequest("Bad range.");
        }

        $project = $this->entityProvider->getByClass(Project::class, $id);

        $this->checkEnabled($project);

        $data = $this->service->get($project, $collapsedGroupIds);
        $range = $this->rangeService->get($data, $scaleUnit, $scale);
        $nonWorkingRanges = $this->getWorkingRanges($range);

        return ResponseComposer::json([
            'total' => $data->total,
            'groups' => array_map(
                /**
                 * @param array{0: ProjectGroup, 1: ?Collection<ProjectTask>, 2: ?int} $item
                 */
                function ($item) {
                    return [
                        'group' => $item[0]->getValueMap(),
                        'tasks' => [
                            'list' => $item[1] ? $item[1]->getValueMapList() : null,
                            'total' => -2,
                        ],
                    ];
                },
                $data->groups,
            ),
            'nonWorkingRanges' => array_map(
                fn ($item) => [$item[0]->toString(), $item[1]->toString()],
                $nonWorkingRanges
            ),
            'range' => [$range[0]->toString(), $range[1]->toString()]
        ]);
    }

    /**
     * @param array{Date, Date} $range
     * @return array{Date, Date}[]
     */
    private function getWorkingRanges(array $range): array
    {
        /** @var array{Date, Date}[] $nonWorkingRanges */
        $nonWorkingRanges = [];

        if ($range[0]->diff($range[1])->days < self::MAX_WORKING_CALENDAR_DAY_RANGE) {
            $nonWorkingRanges = $this->nonWorkingRangesService->get($range);
        }

        return $nonWorkingRanges;
    }

    /**
     * @throws Forbidden
     */
    private function checkEnabled(Project $project): void
    {
        if ($project->hasGanttView()) {
            return;
        }

        throw ForbiddenSilent::createWithBody(
            'ganttViewDisabled',
            Body::create()->withMessageTranslation('ganttViewDisabled', 'Project')
        );
    }
}
