<?php
/***********************************************************************************
 * The contents of this file are subject to the Extension License Agreement
 * ("Agreement") which can be viewed at
 * https://www.espocrm.com/extension-license-agreement/.
 * By copying, installing downloading, or using this file, You have unconditionally
 * agreed to the terms and conditions of the Agreement, and You may not use this
 * file except in compliance with the Agreement. Under the terms of the Agreement,
 * You shall not license, sublicense, sell, resell, rent, lease, lend, distribute,
 * redistribute, market, publish, commercialize, or otherwise transfer rights or
 * usage to the software or any modified version or derivative work of the software
 * created by or for you.
 *
 * Copyright (C) 2024-2025 Letrium Ltd.
 *
 * License ID: f27e70ce6801a13265271f5669c8bc5c
 ************************************************************************************/

namespace Espo\Modules\Project\Hooks\ProjectTask;

use Espo\Core\Hook\Hook\AfterRemove;
use Espo\Core\Hook\Hook\AfterSave;
use Espo\Modules\Project\Entities\ProjectTask;
use Espo\ORM\Entity;
use Espo\ORM\EntityManager;
use Espo\ORM\Query\UpdateBuilder;
use Espo\ORM\Repository\Option\RemoveOptions;
use Espo\ORM\Repository\Option\SaveOptions;

/**
 * @implements AfterSave<ProjectTask>
 * @implements AfterRemove<ProjectTask>
 */
class UpdateSubTaskCount implements AfterSave, AfterRemove
{
    public function __construct(
        private EntityManager $entityManager,
    ) {}

    public function afterSave(Entity $entity, SaveOptions $options): void
    {
        if (!$entity->getParentTaskId()) {
            return;
        }

        if (!$entity->isNew() && !$entity->isAttributeChanged(ProjectTask::ATTR_STATUS)) {
            return;
        }

        $this->update($entity);
    }

    private function update(ProjectTask $entity): void
    {
        $count = $this->entityManager
            ->getRDBRepositoryByClass(ProjectTask::class)
            ->where([
                'parentTaskId' => $entity->getParentTaskId(),
                'status' => [
                    ProjectTask::STATUS_NOT_STARTED,
                    ProjectTask::STATUS_STARTED,
                    ProjectTask::STATUS_DEFERRED,
                ]
            ])
            ->count();

        $update = UpdateBuilder::create()
            ->in(ProjectTask::ENTITY_TYPE)
            ->set(['actualSubTaskCount' => $count])
            ->where(['id' => $entity->getParentTaskId()])
            ->build();

        $this->entityManager->getQueryExecutor()->execute($update);
    }

    public function afterRemove(Entity $entity, RemoveOptions $options): void
    {
        if (!$entity->getParentTaskId()) {
            return;
        }

        $this->update($entity);
    }
}
