<?php
/***********************************************************************************
 * The contents of this file are subject to the Extension License Agreement
 * ("Agreement") which can be viewed at
 * https://www.espocrm.com/extension-license-agreement/.
 * By copying, installing downloading, or using this file, You have unconditionally
 * agreed to the terms and conditions of the Agreement, and You may not use this
 * file except in compliance with the Agreement. Under the terms of the Agreement,
 * You shall not license, sublicense, sell, resell, rent, lease, lend, distribute,
 * redistribute, market, publish, commercialize, or otherwise transfer rights or
 * usage to the software or any modified version or derivative work of the software
 * created by or for you.
 *
 * Copyright (C) 2024-2025 Letrium Ltd.
 *
 * License ID: f27e70ce6801a13265271f5669c8bc5c
 ************************************************************************************/

namespace Espo\Modules\Project\Tools\Project\Cloning;

use Espo\Core\Field\Date;
use Espo\Core\Field\LinkMultiple;
use Espo\Core\Field\LinkMultipleItem;
use Espo\Core\ORM\Repository\Option\SaveOption;
use Espo\Entities\Attachment;
use Espo\Entities\User;
use Espo\Modules\Project\Entities\Project;
use Espo\Modules\Project\Entities\ProjectColumn;
use Espo\Modules\Project\Entities\ProjectGroup;
use Espo\Modules\Project\Entities\ProjectTask;
use Espo\Modules\Project\Hooks\ProjectTask\Prepare;
use Espo\Modules\Project\Tools\ProjectTask\FromTasksLoader;
use Espo\ORM\EntityCollection;
use Espo\ORM\EntityManager;
use Espo\ORM\Query\Part\Order;
use Espo\ORM\Query\SelectBuilder;
use Espo\Repositories\Attachment as AttachmentRepository;
use LogicException;
use RuntimeException;
use SplObjectStorage;

class CloneService
{
    public function __construct(
        private EntityManager $entityManager,
        private User $user,
        private FromTasksLoader $fromTasksLoader,
    ) {}

    public function clone(Project $source, Params $params): Result
    {
        $diffDays = null;

        if ($params->dateStart) {
            $sourceStart = $source->getDateStart();

            if (!$sourceStart) {
                throw new RuntimeException("Cannot clone a project without set Date Start.");
            }

            $diffDays = $this->getDiffDays($params, $sourceStart);
        }

        $cloned = $this->entityManager->getRDBRepositoryByClass(Project::class)->getNew();

        $this->setProjectData($source, $cloned, $params, $diffDays);

        $this->entityManager->saveEntity($cloned);

        $firstColumn = $this->getFirstColumn($cloned);

        $this->cloneMembers($cloned, $source, $params);
        $clonedGroups = $this->cloneGroups($source, $cloned, $params, $diffDays, $firstColumn);

        if ($clonedGroups !== []) {
            $cloned->setActiveGroupId($clonedGroups[0]->getId());

            $this->entityManager->saveEntity($cloned);
        }

        return new Result(id: $cloned->getId());
    }

    private function getDiffDays(Params $params, Date $sourceStart): int
    {
        if (!$params->dateStart) {
            throw new LogicException("No dateStart.");
        }

        $diff = $sourceStart->diff($params->dateStart);

        if ($diff->days === false) {
            throw new RuntimeException("Bad diff.");
        }

        $days = $diff->days;

        if ($diff->invert) {
            $days = -$days;
        }

        return $days;
    }

    private function setProjectData(
        Project $source,
        Project $cloned,
        Params $params,
        ?int $diffDays
    ): void {

        $cloned->setName($params->name);

        if ($diffDays !== null && $source->getDateEnd()) {
            $cloned->setDateEnd($source->getDateEnd()->addDays($diffDays));
        }

        if (!$source->getBoardId()) {
            throw new RuntimeException("No boardId.");
        }

        $cloned
            ->setDateStart($params->dateStart)
            ->setBoardId($source->getBoardId())
            ->setGanttView($source->hasGanttView())
            ->setDescription($source->getDescription());
    }

    private function cloneMembers(Project $cloned, Project $source, Params $params): void
    {
        if (!$params->withMembers) {
            $this->entityManager
                ->getRelation($cloned, Project::RELATION_MEMBERS)
                ->relate($this->user, ['role' => Project::ROLE_OWNER]);

            return;
        }

        /** @var iterable<User> $members */
        $members = $this->entityManager
            ->getRelation($source, Project::RELATION_MEMBERS)
            ->find();

        foreach ($members as $member) {
            $this->entityManager
                ->getRelation($cloned, Project::RELATION_MEMBERS)
                ->relate($member, [
                    'role' => $member->get('projectRole'),
                    'roleId' => $member->get('projectRoleId'),
                ], [SaveOption::SILENT => true]);
        }
    }

    /**
     * @return ProjectGroup[]
     */
    private function cloneGroups(
        Project $source,
        Project $cloned,
        Params $params,
        ?int $diffDays,
        ProjectColumn $firstColumn,
    ): array {

        /** @var SplObjectStorage<ProjectTask, ProjectTask> $sourceClonedMap */
        $sourceClonedMap = new SplObjectStorage();

        /** @var iterable<ProjectGroup> $sourceGroups */
        $sourceGroups = $this->entityManager
            ->getRelation($source, Project::RELATION_GROUPS)
            ->order('order')
            ->find();

        $clonedGroups = [];

        foreach ($sourceGroups as $sourceGroup) {
            $clonedGroup = $this->entityManager->getRDBRepositoryByClass(ProjectGroup::class)->getNew();

            $clonedGroup
                ->setName($sourceGroup->getName())
                ->setColor($sourceGroup->getColor())
                ->setTaskPlacement($sourceGroup->getTaskPlacement())
                ->setOrder($sourceGroup->getOrder())
                ->setProjectId($cloned->getId());

            $this->entityManager->saveEntity($clonedGroup);

            if ($params->withTasks) {
                if ($diffDays === null) {
                    throw new LogicException("No diffDays.");
                }

                $this->cloneGroupTasks(
                    $sourceGroup,
                    $clonedGroup,
                    $params,
                    $diffDays,
                    $firstColumn,
                    $sourceClonedMap
                );
            }

            $clonedGroups[] = $clonedGroup;
        }

        if ($params->withTasks) {
            $this->copyDependencies($sourceClonedMap, $source);
        }

        return $clonedGroups;
    }

    /**
     * @param SplObjectStorage<ProjectTask, ProjectTask> $sourceClonedMap
     */
    private function cloneGroupTasks(
        ProjectGroup $sourceGroup,
        ProjectGroup $clonedGroup,
        Params $params,
        int $diffDays,
        ProjectColumn $firstColumn,
        SplObjectStorage $sourceClonedMap,
    ): void {

        $order = $sourceGroup->getTaskPlacement() === ProjectGroup::TASK_PLACEMENT_BOTTOM ?
            Order::ASC : Order::DESC;

        /** @var iterable<ProjectTask> $sourceTasks */
        $sourceTasks = $this->entityManager
            ->getRelation($sourceGroup, 'tasks')
            ->where(['parentTaskId' => null])
            ->order('order', $order)
            ->find();

        foreach ($sourceTasks as $sourceTask) {
            $clonedTask = $this->entityManager->getRDBRepositoryByClass(ProjectTask::class)->getNew();

            $this->setTaskData($clonedGroup, $sourceTask, $clonedTask, $params, $diffDays, $firstColumn);

            $this->entityManager->saveEntity($clonedTask, [
                SaveOption::SILENT => true,
                Prepare::OPTION_SKIP_ORDER => true,
            ]);

            $sourceClonedMap[$sourceTask] = $clonedTask;

            /** @var iterable<ProjectTask> $sourceSubTasks */
            $sourceSubTasks = $this->entityManager
                ->getRelation($sourceTask, 'subTasks')
                ->order('order', Order::DESC)
                ->find();

            foreach ($sourceSubTasks as $sourceSubTask) {
                $clonedSubTask = $this->entityManager->getRDBRepositoryByClass(ProjectTask::class)->getNew();

                $this->setTaskData($clonedGroup, $sourceSubTask, $clonedSubTask, $params, $diffDays, $firstColumn);
                $clonedSubTask->setParentTaskId($clonedTask->getId());

                $this->entityManager->saveEntity($clonedSubTask, [
                    SaveOption::SILENT => true,
                    Prepare::OPTION_SKIP_ORDER => true,
                ]);

                $sourceClonedMap[$sourceSubTask] = $clonedSubTask;
            }
        }
    }

    private function setTaskData(
        ProjectGroup $clonedGroup,
        ProjectTask $source,
        ProjectTask $cloned,
        Params $params,
        int $diffDays,
        ProjectColumn $column,
    ): void {

        $cloned
            ->setType($source->getType())
            ->setProjectId($clonedGroup->getProjectId())
            ->setGroupId($clonedGroup->getId())
            ->setName($source->getName())
            ->setDescription($source->getDescription())
            ->setHours($source->getHours())
            ->setPoints($source->getPoints())
            ->setPriority($source->getPriority())
            ->setOrder($source->getOrder())
            ->setBoardOrder($source->getBoardOrder())
            ->setColumnId($column->getId());

        if ($source->getDateStart()) {
            $cloned->setDateStart($source->getDateStart()->addDays($diffDays));
        }

        if ($source->getDateEnd()) {
            $cloned->setDateEnd($source->getDateEnd()->addDays($diffDays));
        }

        if ($params->withOwners) {
            $cloned->setOwnerId($source->getOwnerId());
        } else {
            $cloned->setOwnerId($this->user->getId());
        }

        if ($params->withAssignees) {
            $cloned->setAssignedUserId($source->getAssignedUserId());
        }

        if ($params->withAttachments) {
            /** @var iterable<Attachment> $sourceAttachments */
            $sourceAttachments = $this->entityManager
                ->getRelation($source, 'attachments')
                ->find();

            $attachmentIds = [];

            foreach ($sourceAttachments as $sourceAttachment) {
                $clonedAttachment = $this->getAttachmentRepository()->getCopiedAttachment($sourceAttachment);

                $attachmentIds[] = $clonedAttachment->getId();
            }

            if ($attachmentIds !== []) {
                $cloned->set('attachmentsIds', $attachmentIds);
            }
        }
    }

    private function getAttachmentRepository(): AttachmentRepository
    {
        /** @var AttachmentRepository */
        return $this->entityManager->getRepository(Attachment::ENTITY_TYPE);
    }

    private function getFirstColumn(Project $cloned): ProjectColumn
    {
        $firstColumn = $this->entityManager
            ->getRDBRepositoryByClass(ProjectColumn::class)
            ->where(['boardId' => $cloned->getBoardId()])
            ->order('order')
            ->findOne();

        if (!$firstColumn) {
            throw new RuntimeException("No columns in board {$cloned->getBoardId()}.");
        }

        return $firstColumn;
    }

    /**
     * @param SplObjectStorage<ProjectTask, ProjectTask> $sourceClonedMap
     */
    private function loadFromTasks(SplObjectStorage $sourceClonedMap, Project $source): void
    {
        /** @var EntityCollection<ProjectTask> $sourceTasks */
        $sourceTasks = $this->entityManager->getCollectionFactory()->create();

        foreach ($sourceClonedMap as $sourceTask) {
            $sourceTasks[] = $sourceTask;
        }

        $this->fromTasksLoader->load(
            $sourceTasks,
            SelectBuilder::create()
                ->from(ProjectTask::ENTITY_TYPE)
                ->where(['projectId' => $source->getId()])
                ->build()
        );
    }

    /**
     * @param SplObjectStorage<ProjectTask, ProjectTask> $sourceClonedMap
     */
    private function copyDependencies(SplObjectStorage $sourceClonedMap, Project $source): void
    {
        $this->loadFromTasks($sourceClonedMap, $source);

        /** @var array<string, ProjectTask> $idSourceTaskMap */
        $idSourceTaskMap = [];

        foreach ($sourceClonedMap as $sourceTask) {
            $idSourceTaskMap[$sourceTask->getId()] = $sourceTask;
        }

        foreach ($sourceClonedMap as $sourceTask) {
            $sourceLinkMultiple = $sourceTask->getFromTasksLinkMultiple();

            if ($sourceLinkMultiple->getCount() === 0) {
                continue;
            }

            $clonedLinkMultiple = LinkMultiple::create();

            foreach ($sourceLinkMultiple->getList() as $item) {
                $sourceFromTask = $idSourceTaskMap[$item->getId()] ?? null;

                if ($sourceFromTask) {
                    $clonedFromTask = $sourceClonedMap[$sourceFromTask];

                    $clonedLinkMultiple =
                        $clonedLinkMultiple->withAdded(
                            LinkMultipleItem::create($clonedFromTask->getId())
                                ->withColumnValue('type', $item->getColumnValue('type'))
                        );
                }
            }

            if ($clonedLinkMultiple->getCount() === 0) {
                continue;
            }

            $clonedTask = $sourceClonedMap[$sourceTask];

            $clonedTask->setFromTasks($clonedLinkMultiple);
            $this->entityManager->saveEntity($clonedTask, [SaveOption::SILENT => true]);
        }
    }
}
