<?php
/***********************************************************************************
 * The contents of this file are subject to the Extension License Agreement
 * ("Agreement") which can be viewed at
 * https://www.espocrm.com/extension-license-agreement/.
 * By copying, installing downloading, or using this file, You have unconditionally
 * agreed to the terms and conditions of the Agreement, and You may not use this
 * file except in compliance with the Agreement. Under the terms of the Agreement,
 * You shall not license, sublicense, sell, resell, rent, lease, lend, distribute,
 * redistribute, market, publish, commercialize, or otherwise transfer rights or
 * usage to the software or any modified version or derivative work of the software
 * created by or for you.
 *
 * Copyright (C) 2024-2025 Letrium Ltd.
 *
 * License ID: f27e70ce6801a13265271f5669c8bc5c
 ************************************************************************************/

namespace Espo\Modules\Project\Tools\Project\Gantt;

use Espo\Core\Field\Date;
use Espo\Modules\Project\Entities\ProjectTask;
use Espo\Modules\Project\Tools\Project\PlanData;

class RangeService
{
    private const SHIFT_FACTOR = 0.2;

    /**
     * @return array{Date, Date}
     */
    public function get(PlanData $data, string $scaleUnit, int $scale): array
    {
        $start = Date::createToday();
        $end = $start->modify("+$scale $scaleUnit");

        $days = (int) $start->diff($end)->days;
        $dayShift = - floor($days / 2);

        $start = $start->addDays((int) $dayShift);
        $end = $end->addDays((int) $dayShift);

        $minStart = null;
        $maxEnd = null;

        foreach ($data->groups as $group) {
            /** @var iterable<ProjectTask> $tasks */
            $tasks = $group[1] ?? [];

            foreach ($tasks as $task) {
                $minStart ??= $task->getDateStart();
                $maxEnd ??= $task->getDateEnd();

                if ($minStart && $task->getDateStart() && $task->getDateStart()->isLessThan($minStart)) {
                    $minStart = $task->getDateStart();
                }

                if ($maxEnd && $task->getDateEnd() && $task->getDateEnd()->isGreaterThan($maxEnd)) {
                    $maxEnd = $task->getDateEnd();
                }

                if ($maxEnd && $task->getDateStart() && $task->getDateStart()->isGreaterThan($maxEnd)) {
                    $maxEnd = $task->getDateStart();
                }
            }
        }


        if ($minStart && Date::createToday()->isLessThan($minStart)) {
            $start = $minStart;
            $end = $start->modify("+$scale $scaleUnit");
        }

        if ($maxEnd && Date::createToday()->isGreaterThan($maxEnd)) {
            $shiftDays = $this->getShiftDays($scaleUnit);

            $end = $maxEnd->modify("+$shiftDays days");
            $start = $end->modify("-$scale $scaleUnit");
        }

        return [$start, $end];
    }

    private function getShiftDays(string $scaleUnit): int
    {
        $days = Date::createToday()
            ->modify("+1 $scaleUnit")
            ->diff(Date::createToday())
            ->days ?: 0;

        return (int) floor($days * self::SHIFT_FACTOR);
    }
}
