/***********************************************************************************
 * The contents of this file are subject to the Extension License Agreement
 * ("Agreement") which can be viewed at
 * https://www.espocrm.com/extension-license-agreement/.
 * By copying, installing downloading, or using this file, You have unconditionally
 * agreed to the terms and conditions of the Agreement, and You may not use this
 * file except in compliance with the Agreement. Under the terms of the Agreement,
 * You shall not license, sublicense, sell, resell, rent, lease, lend, distribute,
 * redistribute, market, publish, commercialize, or otherwise transfer rights or
 * usage to the software or any modified version or derivative work of the software
 * created by or for you.
 *
 * Copyright (C) 2015-2025 Letrium Ltd.
 *
 * License ID: e4c270586a0c8a9fda53bda910f357e0
 ************************************************************************************/

define(['view'], (/** typeof import('view').default */Dep) => {

    return class BaseReportView extends Dep {

        // language=Handlebars
        templateContent = `
            <div class="row report-control-panel margin-bottom">
                <div class="report-runtime-filters-container col-md-12">{{{runtimeFilters}}}</div>
                <div class="col-md-4 col-md-offset-8">
                    <div class="button-container clearfix">
                        <div class="btn-group pull-right">
                            {{#if hasRuntimeFilters}}
                                <button
                                    class="btn btn-default"
                                    data-action="run"
                                >&nbsp;&nbsp;{{translate 'Run' scope='Report'}}&nbsp;&nbsp;</button>
                            {{else}}
                                <button
                                    class="btn btn-default btn-icon btn-icon-wide"
                                    data-action="refresh"
                                    title="{{translate 'Refresh'}}"
                                ><span class="fas fa-sync-alt"></span></button>
                            {{/if}}
                            {{#if hasDropdownMenu}}
                                <button
                                    type="button"
                                    class="btn btn-default dropdown-toggle"
                                    data-toggle="dropdown"
                                ><span class="fas fa-ellipsis-h"></span></button>
                                <ul class="dropdown-menu">
                                    <li><a
                                        role="button"
                                        tabindex="0"
                                        data-action="exportReport"
                                    >{{translate 'Export'}}</a></li>
                                    {{#if hasSendEmail}}
                                        <li><a
                                            role="button"
                                            tabindex="0"
                                            data-action="sendInEmail"
                                        >{{translate 'Send Email' scope='Report'}}</a></li>
                                    {{/if}}
                                    {{#if hasPrintPdf}}
                                        <li><a
                                            role="button"
                                            tabindex="0"
                                            data-action="printPdf"
                                        >{{translate 'Print to PDF'}}</a></li>
                                    {{/if}}
                                </ul>
                            {{/if}}
                        </div>
                    </div>
                </div>
            </div>
            <div class="hidden information-box text-info margin-bottom small"></div>
            <div class="report-results-container sections-container"></div>
        `

        /**
         * @protected
         * @type {Record}
         */
        result

        /**
         * @protected
         * @type {boolean}
         */
        isPreview

        /**
         * @param {{
         *     model: import('model').default,
         *     isPreview?: boolean,
         * }} options
         */
        constructor(options) {
            super(options);

            this.isPreview = options.isPreview || false;
        }

        data() {
            const data = {
                hasSendEmail: this.getAcl().checkScope('Email'),
                hasRuntimeFilters: this.hasRuntimeFilters(),
                hasPrintPdf: ~(this.getHelper().getAppParam('templateEntityTypeList') || []).indexOf('Report'),
            };

            if (this.isPreview) {
                data.hasSendEmail = false;
                data.hasPrintPdf = false;
            }

            data.hasDropdownMenu = !this.isPreview;

            return data;
        }

        events = {
            /** @this {BaseReportView} */
            'click [data-action="run"]': function () {
                this.refresh();

                this.afterRun();
            },
            /** @this {BaseReportView} */
            'click [data-action="refresh"]': function () {
                this.refresh();
            },
            /** @this {BaseReportView} */
            'click [data-action="exportReport"]': function () {
                this.export();
            },
            /** @this {BaseReportView} */
            'click [data-action="sendInEmail"]': function () {
                this.actionSendInEmail();
            },
            /** @this {BaseReportView} */
            'click [data-action="printPdf"]': function () {
                this.actionPrintPdf();
            },
            /** @this {BaseReportView} */
            'click [data-action="showSubReport"]': function (e) {
                const groupValue = $(e.currentTarget).data('group-value');
                const groupIndex = $(e.currentTarget).data('group-index');

                this.showSubReport(groupValue, groupIndex);
            },
        }

        async refresh() {
            /** @type {HTMLElement} */
            const container = this.element;

            if (container) {
                container.style.minHeight = container.clientHeight + 'px';
            }

            await this.run();

            if (container) {
                container.style.minHeight = '';
            }
        }

        run() {
            return Promise.resolve();
        }

        afterRender() {
            this.$information = this.$el.find('.information-box');
        }

        /**
         *
         * @param {*} groupValue
         * @param {number} groupIndex
         * @param {*} [groupValue2]
         * @param {string} [column]
         */
        showSubReport(groupValue, groupIndex, groupValue2, column) {
            if (this.isPreview) {
                const message = this.translate('noSubReportInPreview', 'messages', 'Report');

                Espo.Ui.warning(message);

                return;
            }

            let reportId = this.model.id;
            let entityType = this.model.get('entityType');

            if (this.result.isJoint) {
                reportId = this.result.columnReportIdMap[column];
                entityType = this.result.columnEntityTypeMap[column];
            }

            this.getCollectionFactory().create(entityType, collection => {
                collection.url = 'Report/action/runList?id=' + reportId + '&groupValue=' +
                    encodeURIComponent(groupValue);

                if (groupIndex) {
                    collection.url += '&groupIndex=' + groupIndex;
                }

                if (groupValue2 !== undefined) {
                    collection.url += '&groupValue2=' + encodeURIComponent(groupValue2);
                }

                if (this.hasRuntimeFilters()) {
                    collection.where = this.lastFetchedWhere;
                }

                collection.maxSize = this.getConfig().get('recordsPerPage') || 20;

                Espo.Ui.notify(' ... ');

                this.createView('subReport', 'advanced:views/report/modals/sub-report', {
                    model: this.model,
                    result: this.result,
                    groupValue: groupValue,
                    collection: collection,
                    groupIndex: groupIndex,
                    groupValue2: groupValue2,
                    column: column,
                }, view => {
                    view.notify(false);
                    view.render();
                });
            });
        }

        initReport() {
            this.once('after:render', () => {
                this.run();
            });

            this.chartType = this.model.get('chartType');

            if (this.hasRuntimeFilters()) {
                this.createRuntimeFilters();
            }
        }

        afterRun() {}

        createRuntimeFilters() {
            const filtersData = this.getStorage().get('state', this.getFilterStorageKey()) || null;

            this.createView('runtimeFilters', 'advanced:views/report/runtime-filters', {
                selector: '.report-runtime-filters-container',
                entityType: this.model.get('entityType'),
                filterList: this.model.get('runtimeFilters'),
                filtersData: filtersData,
            });
        }

        hasRuntimeFilters() {
            if ((this.model.get('runtimeFilters') || []).length) {
                return true;
            }
        }

        getRuntimeFilters() {
            if (this.hasRuntimeFilters()) {
                this.lastFetchedWhere = this.getRuntimeFiltersView().fetch();

                return this.lastFetchedWhere;
            }

            return null;
        }

        getFilterStorageKey() {
            return `report-filters-${this.model.id}`;
        }

        /**
         *
         * @return {ReportRuntimeFiltersView}
         */
        getRuntimeFiltersView() {
            return this.getView('runtimeFilters');
        }

        storeRuntimeFilters() {
            if (this.hasRuntimeFilters()) {
                if (!this.getRuntimeFiltersView()) {
                    return;
                }

                const filtersData = this.getRuntimeFiltersView().fetchRaw();

                this.getStorage().set('state', this.getFilterStorageKey(), filtersData);
            }
        }

        actionSendInEmail() {
            Espo.Ui.notify(' ... ');

            Espo.Ajax
                .postRequest('Report/action/getEmailAttributes', {
                    id: this.model.id,
                    where: this.getRuntimeFilters(),
                }, {timeout: 0})
                .then(attributes => {
                    Espo.Ui.notify(false);

                    this.createView('compose', 'views/modals/compose-email', {
                        attributes: attributes,
                        keepAttachmentsOnSelectTemplate: true,
                        signatureDisabled: true,
                    }, view => {
                        view.render();
                    });
                });
        }

        actionPrintPdf() {
            this.createView('pdfTemplate', 'views/modals/select-template', {entityType: 'Report'}, view => {
                view.render();

                this.listenToOnce(view, 'select', template => {
                    this.clearView('pdfTemplate');

                    const where = this.getRuntimeFilters();

                    const data = {
                        id: this.model.id,
                        where: where,
                        templateId: template.id,
                    };

                    const url = 'Report/action/printPdf';

                    Espo.Ui.notify(' ... ');

                    Espo.Ajax.postRequest(url, data, {timeout: 0}).then(response => {
                        Espo.Ui.notify(false);

                        if ('id' in response) {
                            const url = this.getBasePath() + '?entryPoint=download&id=' + response.id;

                            window.open(url, '_blank');
                        }
                    });
                });
            });
        }

        export() {}
    }
});
