<?php
/***********************************************************************************
 * The contents of this file are subject to the Extension License Agreement
 * ("Agreement") which can be viewed at
 * https://www.espocrm.com/extension-license-agreement/.
 * By copying, installing downloading, or using this file, You have unconditionally
 * agreed to the terms and conditions of the Agreement, and You may not use this
 * file except in compliance with the Agreement. Under the terms of the Agreement,
 * You shall not license, sublicense, sell, resell, rent, lease, lend, distribute,
 * redistribute, market, publish, commercialize, or otherwise transfer rights or
 * usage to the software or any modified version or derivative work of the software
 * created by or for you.
 *
 * Copyright (C) 2024-2025 Letrium Ltd.
 *
 * License ID: f27e70ce6801a13265271f5669c8bc5c
 ************************************************************************************/

namespace Espo\Modules\Project\Classes\Acl\ProjectTask;

use Espo\Core\Acl;
use Espo\Core\Acl\AssignmentChecker as AssignmentCheckerInterface;
use Espo\Entities\User;
use Espo\Modules\Project\Entities\Project;
use Espo\Modules\Project\Entities\ProjectRole;
use Espo\Modules\Project\Entities\ProjectTask;
use Espo\Modules\Project\Tools\Project\MemberRoleProvider;
use Espo\ORM\Entity;
use Espo\ORM\EntityManager;

/**
 * @implements AssignmentCheckerInterface<ProjectTask>
 */
class AssignmentChecker implements AssignmentCheckerInterface
{
    private const ATTR_ASSIGNED_USER_ID = 'assignedUserId';

    public function __construct(
        private EntityManager $entityManager,
        private MemberRoleProvider $memberRoleProvider,
        private Acl $acl,
    ) {}

    public function check(User $user, Entity $entity): bool
    {
        if (!$this->checkAssignedUser($user, $entity)) {
            return false;
        }

        return true;
    }

    private function checkAssignedUser(User $user, ProjectTask $entity): bool
    {
        if (!$entity->getProjectId()) {
            return false;
        }

        if (!$entity->getAssignedUserId()) {
            return true;
        }

        if (!$entity->isAttributeChanged(self::ATTR_ASSIGNED_USER_ID)) {
            return true;
        }

        $project = $this->entityManager->getRDBRepositoryByClass(Project::class)->getById($entity->getProjectId());

        if (!$project) {
            return false;
        }

        $isRelated = $this->entityManager
            ->getRelation($project, 'members')
            ->isRelatedById($entity->getAssignedUserId());

        if (!$isRelated) {
            return false;
        }

        if ($user->isAdmin()) {
            return true;
        }

        $memberRole = $this->memberRoleProvider->get($user, $project->getId());

        if (!$memberRole) {
            return false;
        }

        if (
            $this->acl->checkEntityEdit($project) ||
            $memberRole->record->getTaskAssignment() === ProjectRole::LEVEL_ALL
        ) {
            return true;
        }

        return $entity->getAssignedUserId() === $user->getId();
    }
}
