<?php
/***********************************************************************************
 * The contents of this file are subject to the Extension License Agreement
 * ("Agreement") which can be viewed at
 * https://www.espocrm.com/extension-license-agreement/.
 * By copying, installing downloading, or using this file, You have unconditionally
 * agreed to the terms and conditions of the Agreement, and You may not use this
 * file except in compliance with the Agreement. Under the terms of the Agreement,
 * You shall not license, sublicense, sell, resell, rent, lease, lend, distribute,
 * redistribute, market, publish, commercialize, or otherwise transfer rights or
 * usage to the software or any modified version or derivative work of the software
 * created by or for you.
 *
 * Copyright (C) 2024-2025 Letrium Ltd.
 *
 * License ID: f27e70ce6801a13265271f5669c8bc5c
 ************************************************************************************/

namespace Espo\Modules\Project\Tools\ProjectTask;

use Espo\Core\Acl;
use Espo\Core\Exceptions\Error\Body;
use Espo\Core\Exceptions\Forbidden;
use Espo\Core\Exceptions\NotFound;
use Espo\Core\Field\Link;
use Espo\Modules\Project\Entities\Project;
use Espo\Modules\Project\Entities\ProjectColumn;
use Espo\Modules\Project\Entities\ProjectGroup;
use Espo\Modules\Project\Entities\ProjectTask;
use Espo\ORM\EntityManager;
use Espo\ORM\Query\Part\Expression as Expr;
use Espo\ORM\Query\Part\Order;
use Espo\ORM\Query\UpdateBuilder;
use Espo\ORM\Repository\RDBRelation;
use RuntimeException;
use Traversable;

class RecordService
{
    public const SAVE_OPTION_SKIP_SET_STATUS = 'skipSetStatus';

    public const PLACEMENT_TOP = 'top';
    public const PLACEMENT_BOTTOM = 'bottom';

    public function __construct(
        private EntityManager $entityManager,
        private Acl $acl,
    ) {}

    /**
     * @throws Forbidden
     */
    public function complete(ProjectTask $task): void
    {
        $column = $this->getColumn($task, ProjectTask::STATUS_COMPLETED);

        $this->checkAlreadyCompleted($task, $column);
        $this->checkCompleteDependency($task);
        $this->checkCompleteSubTasks($task);

        $task->setStatus(ProjectTask::STATUS_COMPLETED);
        $task->setColumnLink($column ? Link::create($column->getId(), $column->getName()) : null);

        $this->entityManager->saveEntity($task, [self::SAVE_OPTION_SKIP_SET_STATUS => true]);
    }

    /**
     * @throws Forbidden
     */
    public function cancel(ProjectTask $task): void
    {
        $column = $this->getColumn($task, ProjectTask::STATUS_CANCELED);

        $this->checkAlreadyCanceled($task, $column);

        $task->setStatus(ProjectTask::STATUS_CANCELED);
        $task->setColumnLink($column ? Link::create($column->getId(), $column->getName()) : null);

        $this->entityManager->saveEntity($task, [self::SAVE_OPTION_SKIP_SET_STATUS => true]);
    }

    /**
     * @throws Forbidden
     */
    public function checkCompleteDependency(ProjectTask $task): void
    {
        if ($task->isNew()) {
            // @todo Fetch dependent tasks by ids.
            return;
        }

        /** @var RDBRelation<ProjectTask> $relation */
        $relation = $this->entityManager->getRelation($task, 'fromTasks');

        $notStarted = $relation
            ->where([
                '@relation.type' => ProjectTask::DEPENDENCY_TYPE_SF,
                'status' => [
                    ProjectTask::STATUS_NOT_STARTED,
                    ProjectTask::STATUS_CANCELED,
                    ProjectTask::STATUS_DEFERRED,
                ],
            ])
            ->order('order')
            ->findOne();

        if ($notStarted) {
            throw Forbidden::createWithBody(
                'cannotCompleteFromTaskNotStarted',
                Body::create()
                    ->withMessageTranslation('cannotCompleteFromTaskNotStarted', ProjectTask::ENTITY_TYPE, [
                        'fromId' => $notStarted->getId(),
                        'fromName' => $notStarted->getName(),
                    ])
            );
        }

        $notCompleted = $relation
            ->where([
                '@relation.type' => ProjectTask::DEPENDENCY_TYPE_FF,
                'status!=' => [
                    ProjectTask::STATUS_COMPLETED,
                ],
            ])
            ->order('order')
            ->findOne();

        if ($notCompleted) {
            throw Forbidden::createWithBody(
                'cannotCompleteFromTaskNotCompleted',
                Body::create()
                    ->withMessageTranslation('cannotCompleteFromTaskNotCompleted', ProjectTask::ENTITY_TYPE, [
                        'fromId' => $notCompleted->getId(),
                        'fromName' => $notCompleted->getName(),
                    ])
            );
        }
    }

    /**
     * @throws Forbidden
     */
    public function checkCompleteSubTasks(ProjectTask $task): void
    {
        if ($task->isNew()) {
            return;
        }

        $one = $this->entityManager
            ->getRelation($task, 'subTasks')
            ->where([
                'status!=' => [
                    ProjectTask::STATUS_COMPLETED,
                    ProjectTask::STATUS_CANCELED,
                ],
            ])
            ->findOne();

        if (!$one) {
            return;
        }

        throw Forbidden::createWithBody(
            'cannotCompleteSubTaskNotCompleted',
            Body::create()
                ->withMessageTranslation('cannotCompleteSubTaskNotCompleted', ProjectTask::ENTITY_TYPE)
        );
    }

    /**
     * @throws Forbidden
     */
    public function moveInGroup(ProjectTask $task, string $placement): void
    {
        $groupId = $task->getGroupId();

        if (!$groupId) {
            throw new Forbidden("No group.");
        }

        $top = $placement === self::PLACEMENT_TOP;

        $one = $this->entityManager
            ->getRDBRepositoryByClass(ProjectTask::class)
            ->where(['groupId' => $groupId])
            ->order('order', $top ? Order::ASC : Order::DESC)
            ->findOne();

        $order = 0;

        if ($one) {
            $order = $top ?
                $one->getOrder() - 10 :
                $one->getOrder() + 10;
        }

        $task->setOrder($order);
        $this->entityManager->saveEntity($task);
    }

    /**
     * @todo Call mass action.
     * @throws Forbidden
     */
    public function checkStartDependency(ProjectTask $task): void
    {
        /** @var RDBRelation<ProjectTask> $relation */
        $relation = $this->entityManager->getRelation($task, 'fromTasks');

        $notStarted = $relation
            ->where([
                '@relation.type' => ProjectTask::DEPENDENCY_TYPE_SS,
                'status' => [
                    ProjectTask::STATUS_NOT_STARTED,
                    ProjectTask::STATUS_CANCELED,
                    ProjectTask::STATUS_DEFERRED,
                ],
            ])
            ->order('order')
            ->findOne();

        if ($notStarted) {
            throw Forbidden::createWithBody(
                'cannotStartFromTaskNotStarted',
                Body::create()
                    ->withMessageTranslation('cannotStartFromTaskNotStarted', ProjectTask::ENTITY_TYPE, [
                        'fromId' => $notStarted->getId(),
                        'fromName' => $notStarted->getName(),
                    ])
            );
        }

        $notCompleted = $relation
            ->where([
                '@relation.type' => ProjectTask::DEPENDENCY_TYPE_FS,
                'status!=' => ProjectTask::STATUS_COMPLETED,
            ])
            ->order('order')
            ->findOne();

        if ($notCompleted) {
            throw Forbidden::createWithBody(
                'cannotStartFromTaskNotCompleted',
                Body::create()
                    ->withMessageTranslation('cannotStartFromTaskNotCompleted', ProjectTask::ENTITY_TYPE, [
                        'fromId' => $notCompleted->getId(),
                        'fromName' => $notCompleted->getName(),
                    ])
            );
        }
    }

    private function getColumn(ProjectTask $task, string $status): ?ProjectColumn
    {
        if (!$task->getProjectId()) {
            return null;
        }

        $project = $this->entityManager->getRDBRepositoryByClass(Project::class)->getById($task->getProjectId());

        if (!$project || !$project->getBoardId()) {
            return null;
        }

        return $this->entityManager
            ->getRDBRepositoryByClass(ProjectColumn::class)
            ->order('order')
            ->where([
                'boardId' => $project->getBoardId(),
                'mappedStatus' => $status,
            ])
            ->findOne();
    }

    /**
     * @throws Forbidden
     */
    private function checkAlreadyCompleted(ProjectTask $task, ?ProjectColumn $column): void
    {
        if (
            $task->getStatus() === ProjectTask::STATUS_COMPLETED &&
            $task->getColumnId() === $column?->getId()
        ) {
            throw Forbidden::createWithBody(
                'cannotCompleteAlreadyCompleted',
                Body::create()->withMessageTranslation('cannotCompleteAlreadyCompleted', ProjectTask::ENTITY_TYPE)
            );
        }
    }

    /**
     * @throws Forbidden
     */
    private function checkAlreadyCanceled(ProjectTask $task, ?ProjectColumn $column): void
    {
        if (
            $task->getStatus() === ProjectTask::STATUS_CANCELED &&
            $task->getColumnId() === $column?->getId()
        ) {
            throw Forbidden::createWithBody(
                'cannotCompleteAlreadyCanceled',
                Body::create()->withMessageTranslation('cannotCompleteAlreadyCanceled', ProjectTask::ENTITY_TYPE)
            );
        }
    }

    /**
     * @throws NotFound
     */
    public function changeColumn(ProjectTask $task, string $columnId): void
    {
        $boardId = $task->getBoardId();

        if (!$boardId) {
            throw new RuntimeException("No boardId.");
        }

        $column = $this->entityManager
            ->getRDBRepositoryByClass(ProjectColumn::class)
            ->where([
                'id' => $columnId,
                'boardId' => $boardId,
            ])
            ->findOne();

        if (!$column) {
            throw new NotFound("Column not found.");
        }

        $status = $column->getMappedStatus();

        $task->setStatus($status);
        $task->setColumnLink(Link::create($columnId, $column->getName()));

        $this->entityManager->saveEntity($task);
    }

    /**
     * @param string[] $ids
     * @throws Forbidden
     */
    public function massMoveToGroup(array $ids, ProjectGroup $group): void
    {
        /** @var iterable<ProjectTask> $tasks */
        $tasks = $this->entityManager
            ->getRDBRepositoryByClass(ProjectTask::class)
            ->where(['id' => $ids])
            ->order('order')
            ->find();

        foreach ($tasks as $task) {
            if (!$this->acl->checkEntityEdit($task)) {
                throw Forbidden::createWithBody(
                    'noEditAccess',
                    Body::create()->withMessageTranslation('noEditAccess', ProjectTask::ENTITY_TYPE)
                );
            }
        }

        foreach ($tasks as $task) {
            $task->setGroup(Link::create($group->getId(), $group->getName()));

            $this->entityManager->saveEntity($task);
        }
    }

    /**
     * @param string[] $ids
     * @throws Forbidden
     */
    public function massReOrder(array $ids, ProjectTask $targetTask, ProjectGroup $group, bool $isAfter): void
    {
        if ($targetTask->getParentTaskId()) {
            throw new Forbidden("Task is sub-task.");
        }

        $step = 10;

        $this->entityManager->getTransactionManager()->start();

        /** @var Traversable<ProjectTask>&iterable<ProjectTask> $tasks */
        $tasks = $this->entityManager
            ->getRDBRepositoryByClass(ProjectTask::class)
            ->forUpdate()
            ->where([
                'id' => $ids,
                'parentTaskId' => null,
            ])
            ->order('order')
            ->find();

        foreach ($tasks as $task) {
            if ($task->getGroupId() !== $group->getId()) {
                $this->entityManager->getTransactionManager()->rollback();

                throw new Forbidden("Task is in another group.");
            }

            if ($task->getParentTaskId()) {
                $this->entityManager->getTransactionManager()->rollback();

                throw new Forbidden("Task is sub-task.");
            }
        }

        $count = count(iterator_to_array($tasks));
        $diff = $count * $step;
        $fromOrder = $targetTask->getOrder();

        $lastOrder = $fromOrder;

        foreach ($tasks as $i => $task) {
            $order = $fromOrder + $i  * $step;

            if ($isAfter) {
                $order += $step;
            }

            $task->setOrder($order);
            $this->entityManager->saveEntity($task);

            $lastOrder = $task->getOrder();
        }

        if (!$isAfter) {
            $targetTask->setOrder($lastOrder + $step);
            $this->entityManager->saveEntity($targetTask);

            $diff += $step;
        }

        $update = UpdateBuilder::create()
            ->in(ProjectTask::ENTITY_TYPE)
            ->where([
                'groupId' => $group->getId(),
                'order>=' => $fromOrder,
                'parentTaskId' => null,
                'id!=' => [...$ids, $targetTask->getId()],
            ])
            ->set([
               'order' => Expr::add(Expr::column('order'), $diff),
            ])
            ->build();

        $this->entityManager->getQueryExecutor()->execute($update);
        $this->entityManager->getTransactionManager()->commit();
    }

    /**
     * @param string[] $ids
     * @throws Forbidden
     */
    public function massSubTaskReOrder(
        array $ids,
        ProjectTask $targetTask,
        ProjectTask $parentTask,
        bool $isAfter
    ): void {

        if ($targetTask->getParentTaskId() !== $parentTask->getId()) {
            throw new Forbidden("Task is not sub-task of parent.");
        }

        $step = 10;

        $this->entityManager->getTransactionManager()->start();

        /** @var Traversable<ProjectTask>&iterable<ProjectTask> $tasks */
        $tasks = $this->entityManager
            ->getRDBRepositoryByClass(ProjectTask::class)
            ->forUpdate()
            ->where([
                'id' => $ids,
                'parentTaskId' => $targetTask->getParentTaskId(),
            ])
            ->order('order')
            ->find();

        foreach ($tasks as $task) {
            if ($task->getParentTaskId() !== $parentTask->getId()) {
                $this->entityManager->getTransactionManager()->rollback();

                throw new Forbidden("Task is in another parent.");
            }
        }

        $count = count(iterator_to_array($tasks));
        $diff = $count * $step;
        $fromOrder = $targetTask->getOrder();

        $lastOrder = $fromOrder;

        foreach ($tasks as $i => $task) {
            $order = $fromOrder + $i  * $step;

            if ($isAfter) {
                $order += $step;
            }

            $task->setOrder($order);
            $this->entityManager->saveEntity($task);

            $lastOrder = $task->getOrder();
        }

        if (!$isAfter) {
            $targetTask->setOrder($lastOrder + $step);
            $this->entityManager->saveEntity($targetTask);

            $diff += $step;
        }

        $update = UpdateBuilder::create()
            ->in(ProjectTask::ENTITY_TYPE)
            ->where([
                'order>=' => $fromOrder,
                'parentTaskId' => $parentTask->getId(),
                'id!=' => [...$ids, $targetTask->getId()],
            ])
            ->set([
                'order' => Expr::add(Expr::column('order'), $diff),
            ])
            ->build();

        $this->entityManager->getQueryExecutor()->execute($update);
        $this->entityManager->getTransactionManager()->commit();
    }
}
