/***********************************************************************************
 * The contents of this file are subject to the Extension License Agreement
 * ("Agreement") which can be viewed at
 * https://www.espocrm.com/extension-license-agreement/.
 * By copying, installing downloading, or using this file, You have unconditionally
 * agreed to the terms and conditions of the Agreement, and You may not use this
 * file except in compliance with the Agreement. Under the terms of the Agreement,
 * You shall not license, sublicense, sell, resell, rent, lease, lend, distribute,
 * redistribute, market, publish, commercialize, or otherwise transfer rights or
 * usage to the software or any modified version or derivative work of the software
 * created by or for you.
 *
 * Copyright (C) 2015-2025 Letrium Ltd.
 *
 * License ID: e4c270586a0c8a9fda53bda910f357e0
 ************************************************************************************/

define(['advanced:views/report/reports/base'], (/** typeof BaseReportView */Dep) => {

    return class Grid2ReportView extends Dep {

        setup() {
            this.initReport();
        }

        export() {
            const where = this.getRuntimeFilters();

            const columnsTranslation = {};
            const entityType = this.model.get('entityType');

            const columnList = (this.model.get('columns') || []).filter(item => {
                return this.options.reportHelper.isColumnSummary(item);
            });

            columnList.forEach(item => {
                columnsTranslation[item] = this.options.reportHelper.translateGroupName(item, entityType);
            });

            const o = {
                scope: entityType,
                reportType: 'Grid',
                columnList: columnList,
                columnsTranslation: columnsTranslation,
            };

            let url;

            const data = {
                id: this.model.id,
                where: where,
            };

            this.createView('dialogExport', 'advanced:views/report/modals/export-grid', o, view => {
                view.render();

                this.listenToOnce(view, 'proceed', dialogData => {
                    data.column = dialogData.column;

                    if (dialogData.format === 'csv') {
                        url = 'Report/action/exportGridCsv';
                        data.column = dialogData.column;
                    } else if (dialogData.format === 'xlsx') {
                        url = 'Report/action/exportGridXlsx';
                    }

                    Espo.Ui.notify(' ... ');

                    Espo.Ajax.postRequest(url, data, {timeout: 0}).then(response => {
                        Espo.Ui.notify(false);

                        if ('id' in response) {
                            window.location = this.getBasePath() + '?entryPoint=download&id=' + response.id;
                        }
                    });
                });
            });
        }

        async run() {
            Espo.Ui.notify(' ... ');

            const $container = this.$el.find('.report-results-container');
            $container.empty();

            const where = this.getRuntimeFilters();

            /** @type {Record} */
            let result;

            const data = this.isPreview ? this.model.attributes : null;

            if (this.isPreview) {
                /** @type {Record} */
                result = await Espo.Ajax.postRequest('Report/runGridPreview', {
                    where: where,
                    data: data,
                }, {timeout: 0});
            } else {
                /** @type {Record} */
                result = await Espo.Ajax.getRequest('Report/action/run', {
                    id: this.model.id,
                    where: where,
                    data: data,
                }, {timeout: 0});
            }

            Espo.Ui.notify(false);

            this.result = result;

            this.storeRuntimeFilters();

            this.processInformation();

            const headerTag = this.options.isLargeMode ? 'h4' : 'h5';
            const headerMarginTop = this.options.isLargeMode ? 60 : 50;

            const summaryColumnList = result.summaryColumnList || result.columnList;

            summaryColumnList.forEach((column, i) => {
                const $column = $('<div>')
                    .addClass('column-' + i)
                    .addClass('section')
                    .addClass('sections-container');

                const $header = $('<' + headerTag + ' style="margin-bottom: 25px">' +
                    this.options.reportHelper.formatColumn(column, result) + '</' + headerTag + '>');

                if (!this.options.isLargeMode) {
                    $header.addClass('text-soft');
                }

                if (headerMarginTop && i) {
                    $header.css('marginTop', headerMarginTop);
                }

                const $tableContainer = $('<div>')
                    .addClass('report-table clearfix')
                    .addClass('report-table-' + i)
                    .addClass('section');

                const $chartContainer = $('<div>')
                    .addClass('report-chart')
                    .addClass('report-chart-' + i)
                    .addClass('section');

                if (this.chartType) {
                    $tableContainer.addClass('margin-bottom');
                }

                $column.append($header);

                if (!this.options.showChartFirst) {
                    $column.append($tableContainer);
                }

                if (this.chartType) {
                    $column.append($chartContainer);
                }

                if (this.options.showChartFirst) {
                    $column.append($tableContainer);
                }

                $container.append($column);
            });

            summaryColumnList.forEach((column, i) => {
                this.createView(`reportTable${i}`, 'advanced:views/report/reports/tables/grid2', {
                    selector:  `.report-results-container .column-${i} .report-table`,
                    column: column,
                    result: result,
                    reportHelper: this.options.reportHelper,
                    hasChart: !!this.chartType,
                    isLargeMode: this.options.isLargeMode,
                    showChartFirst: this.options.showChartFirst,
                }, view => {
                    view.render();
                });

                if (this.chartType) {
                    const viewName = 'advanced:views/report/reports/charts/grid2' +
                        Espo.Utils.camelCaseToHyphen(this.chartType);

                    this.createView('reportChart' + i, viewName, {
                        selector: `.report-results-container .column-${i} .report-chart`,
                        column: column,
                        result: result,
                        reportHelper: this.options.reportHelper,
                        colors: result.chartColors || {},
                        color: result.chartColor || null,
                    }, (view) => {
                        view.render();

                        this.listenTo(view, 'click-group', (groupValue, groupIndex, groupValue2) => {
                            this.showSubReport(groupValue, groupIndex, groupValue2);
                        });
                    });
                }
            });
        }

        processInformation() {
            if (this.result.emptyStringGroupExcluded) {
                this.$information
                    .removeClass('hidden')
                    .text(this.translate('emptyStringGroupExcluded', 'messages', 'Report'));

                return;
            }

            this.$information
                .addClass('hidden')
                .text('');
        }
    }
})
