<?php
/*
 * This file is part of  the extension: Ebla Public Form
 * Copyright (c) Eblasoft Bilişim Ltd.
 *
 * This Software is the property of Eblasoft Bilişim Ltd. and is protected
 * by copyright law - it is NOT Freeware and can be used only in one project
 * under a proprietary license, which is delivered along with this program.
 * If not, see <http://eblasoft.com.tr/eula>.
 *
 * This Software is distributed as is, with LIMITED WARRANTY AND LIABILITY.
 * Any unauthorised use of this Software without a valid license is
 * a violation of the License Agreement.
 *
 * According to the terms of the license you shall not resell, sublicense,
 * rent, lease, distribute or otherwise transfer rights or usage of this
 * Software or its derivatives. You may modify the code of this Software
 * for your own needs, if source code is provided.
 */

namespace Espo\Modules\EblaForm\Services;

use Espo\Core\DataManager;
use Espo\Core\Exceptions\BadRequest;
use Espo\Core\Exceptions\Conflict;
use Espo\Core\Exceptions\Error;
use Espo\Core\Exceptions\Forbidden;
use Espo\Core\Exceptions\ForbiddenSilent;
use Espo\Core\Exceptions\NotFound;
use Espo\Core\Record\CreateParams;
use Espo\Core\Templates\Services\Base;
use Espo\Core\Utils\File\Manager as FileManager;
use Espo\Core\Utils\Json;
use Espo\Core\Utils\Language;
use Espo\Core\Utils\Log;
use Espo\Core\Utils\Resource\FileReader;
use Espo\Core\Utils\Util;
use Espo\Modules\EblaForm\Core\Helper;
use Espo\ORM\Entity;
use Espo\Tools\Layout\LayoutProvider;
use Espo\Tools\Layout\Service;
use Exception;
use JsonException;
use stdClass;


class EblaForm extends Base
{
    protected $forceSelectAllAttributes = true;
    protected $layout;
    protected $language;
    private $helper;
    private $fileReader;
    private $layoutService;
    private $dataManager;

    public function __construct(
        LayoutProvider $layout,
        Language $language,
        Helper $helper,
        Service $layoutService,
        FileReader $fileReader,
        DataManager $dataManager,
        private FileManager $fileManager,
        private Log $log
    ) {
        parent::__construct();
        $this->layout = $layout;
        $this->language = $language;
        $this->helper = $helper;
        $this->fileReader = $fileReader;
        $this->layoutService = $layoutService;
        $this->dataManager = $dataManager;
    }

    /**
     * @throws Forbidden
     * @throws JsonException
     */
    public function getEblaFormDefs(string $id = null): stdClass
    {
        $eblaForm = $this->getEntity($id);

        if (!$eblaForm->get('isActive')) {
            throw new Forbidden();
        }

        $scope = $eblaForm->get('entity');
        $layoutName = $eblaForm->get('layout') ?? 'detail';
        $i18n = $this->language->getAll();

        $layoutStr = $this->layout->get($scope, $layoutName);
        if (!$layoutStr) {
            $layoutStr = $this->layout->get($scope, 'detail');
        }

        $addEntityDefs = [];
        $fields = $this->metadata->get(['entityDefs', $scope, 'fields']);
        foreach ($fields as $field) {
            if (in_array($field['type'], ['enum', 'multiEnum', 'array', 'checklist']) && !empty($field['optionsReference'])) {
                $ref = explode('.', $field['optionsReference']);
                $addEntityDefs[$ref[0]]['fields'][$ref[1]]['options'] = $this->metadata->get(['entityDefs', $ref[0], 'fields', $ref[1], 'options']);
            }
        }

        return (object)[
            'scope' => $scope,
            'showLogo' => $eblaForm->get('showLogo'),
            'welcomeMessage' => $eblaForm->get('welcomeMessage'),
            'thanksMessage' => $eblaForm->get('thanksMessage'),
            'defaultValues' => $eblaForm->get('defaultValues'),
            'detailLayout' => Json::decode($layoutStr) ?? [],
            'i18n' => [
                'Global' => [
                    'sets' => $i18n['Global']['sets'],
                    'labels' => $i18n['Global']['labels'],
                    'fields' => $i18n['Global']['fields'],
                    'links' => $i18n['Global']['links'],
                    'messages' => $i18n['Global']['messages'],
                    'options' => $i18n['Global']['options'],
                    'lists' => $i18n['Global']['lists'],
                    'durationUnits' => $i18n['Global']['durationUnits'],
                ],
                $scope => $i18n[$scope],
            ],
            'metadata' => [
                'fields' => $this->metadata->get(['fields']),
                'entityDefs' => [
                    ...$addEntityDefs,
                    $scope => $this->metadata->get(['entityDefs', $scope])
                ],
                'clientDefs' => [$scope => $this->metadata->get(['clientDefs', $scope])],
                'scopes' => [$scope => $this->metadata->get(['scopes', $scope])],
                'app' => ['regExpPatterns' => $this->metadata->get(['app', 'regExpPatterns'])],
            ]
        ];
    }

    /**
     * @throws Forbidden
     * @throws ForbiddenSilent
     * @throws BadRequest
     */
    public function saveEblaFormForm(stdClass $data, string $eblaFormId): stdClass
    {
        $eblaForm = $this->getEntity($eblaFormId);

        if (!$eblaForm->get('isActive')) {
            throw new Forbidden();
        }

        $integration = $this->entityManager->getEntity('Integration', 'ReCaptcha');

        if ($eblaForm->get('reCaptchaEnabled') && $integration && $integration->get('enabled')) {
            $reCaptchaToken = $data->reCaptchaToken;

            if (!$reCaptchaToken) {
                throw new BadRequest("ReCaptcha token is empty or expired");
            }

            $secret = $integration->get('reCaptchaSecretKey');

            $url = "https://www.google.com/recaptcha/api/siteverify?secret=$secret&response=$reCaptchaToken";
            $request = $this->curlGetFileContents($url);
            $result = json_decode($request, true);
            if (!$result['success']) {
                $this->log->error("ReCaptcha Error: " . json_encode($result));
                throw new BadRequest("ReCaptcha Check Error");
            }
        }

        $scope = $eblaForm->get('entity');
        $entity = $this->entityManager->getEntity($scope);
        $entity->set($data);
        $entity->set($eblaForm->get('defaultValues')->defaultValues);
        $this->entityManager->saveEntity($entity);

        return $entity->getValueMap();
    }

    function curlGetFileContents($url)
    {
        $c = curl_init();
        curl_setopt($c, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($c, CURLOPT_URL, $url);
        $contents = curl_exec($c);
        curl_close($c);

        if ($contents)
            return $contents;
        else return FALSE;
    }

    /**
     * @throws BadRequest
     * @throws Forbidden
     * @throws Conflict
     */
    public function eblaUpload(StdClass $data, CreateParams $params, $eblaFormId): Entity
    {
        $eblaForm = $this->getEntity($eblaFormId);

        if (!$eblaForm->get('isActive')) {
            throw new Forbidden();
        }

        if ($this->config->get('allowEblaFormUpload') !== true) {
            //throw new Forbidden();
        }
        $attachmentService = $this->recordServiceContainer->get('Attachment');
        return $attachmentService->create($data, $params);
    }

    public function eblaFormJob($jobData): void
    {
        $info = $this->helper->getInfo();

        if (empty($info)) {
            return;
        }

        $data = array(
            'lid' => @$info['lid'],
            'name' => 'Ebla Public Form',
            'site' => $this->config->get('siteUrl'),
            'version' => '6.9.1',
            'installedAt' => @$info['installedAt'],
            'updatedAt' => @$info['created_at'],
            'applicationName' => $this->config->get('applicationName'),
            'espoVersion' => $this->config->get('version'),
        );

        try {
            $result = $this->validate($data);

            if ($result) {
                $this->helper->setInfo($info, $result);
            }
        }
        catch (Exception $e) {
            $GLOBALS['log']->debug('Ebla Form: ' . $e->getMessage());
        }
    }

    protected function validate(array $data)
    {
        if (function_exists('curl_version')) {
            $ch = curl_init();

            $payload = json_encode($data);
            curl_setopt($ch, CURLOPT_URL, base64_decode('aHR0cHM6Ly9jcm0uZWJsYXNvZnQuY29tLnRyL2FwaS92MS92YWxpZGF0ZQ=='));
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 60);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Content-Type: application/json',
                'Content-Length: ' . strlen($payload)
            ]);

            $curlVersion = curl_version();
            $curlVersion = explode('.', $curlVersion['version']);
            $curlVersion = $curlVersion[0] . '.' . $curlVersion[1];

            if ($curlVersion > 7.3) {
                curl_setopt($ch, CURLOPT_PROXY_SSL_VERIFYPEER, false);
            }

            $result = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);

            if ($httpCode === 200) {
                return json_decode($result, true);
            }
        }

        return false;
    }

    /**
     * @throws Error
     * @throws NotFound
     */
    protected function afterCreateEntity(Entity $entity, $data)
    {
        parent::afterCreateEntity($entity, $data);

        if ($entity->get('layout') === 'createNewLayout') {
            $this->addLayout($entity);
        }
    }

    /**
     * @throws NotFound
     * @throws Error
     */
    public function addLayout($entity)
    {
        $layoutManager = new LayoutProvider(
            $this->fileManager,
            $this->injectableFactory,
            $this->metadata,
            $this->fileReader
        );

        $type = 'detail';

        $scope = $this->sanitizeLayoutInput($entity->get('entity'));
        $layoutName = $this->sanitizeLayoutInput($entity->get('name'));

        if (empty($scope) || empty($layoutName)) {
            return false;
        }

        $layoutName = str_replace(' ', '-', strtolower($layoutName));
        $layoutName = Util::hyphenToCamelCase($layoutName);

        $result = $this->layoutService->update($scope, $layoutName, null, json_decode($layoutManager->get($scope, $type)));
        if ($result) {
            $entity->set('layout', $layoutName);
            $this->entityManager->saveEntity($entity);

            $clientDefs = $this->metadata->get(['clientDefs', $scope]);

            if (!array_key_exists('additionalLayouts', $clientDefs)) {
                $clientDefs['additionalLayouts'] = [];
            }

            $clientDefs['additionalLayouts'][$layoutName] = [
                'type' => $type
            ];

            $this->metadata->set('clientDefs', $scope, $clientDefs);
            $this->metadata->save();

            $this->language->set('Admin', 'layouts', $layoutName, ucfirst($entity->get('name') . ' (Public Form)'));
            $this->language->save();
            $this->dataManager->updateCacheTimestamp();

            return json_decode($layoutManager->get($scope, $layoutName));
        } else {
            throw new Error("Error while saving layout.");
        }
    }

    protected function sanitizeLayoutInput(string $name): string
    {
        return preg_replace("([.]{2,})", '', $name);
    }
}
